﻿/**************************************************************
*               Copyright (C) 2011 by L-1 Identity Solutions
*               All Rights Reserved.
*
*  Name:        MainForm.cs
*  Author:      John MacDonald
*  Description: TouchPrint API C# Sample Firmware Update Application
*
* This software is and contains proprietary, confidential, and trade secret
* property of Identix Corporation.  Disclosure or distribution without
* written permission of Identix Corporation may be subject to civil or
* criminal prosecution.
*/
using System;
using System.IO;
using System.Threading;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.Xml;
using System.Xml.Schema;
using System.Xml.Serialization;
using TPAPI.NetWrapper;

namespace FirmwareUpdate
{
   public partial class MainForm : Form
   {
      private const uint MAX_DEVLIST = 4;
      private const uint TPAPI_LNK_USB = 0;
      private const uint TPAPI_LNK_1394_OHCI = 1;
      private const uint TPAPI_LNK_1394_UB = 2;
      private bool bUpdateInProgress = false;
      private TpApi tpApi = new TpApi();
      private TpApiDevice[] tpDevices;
      private TpApiDevice tpDevice = null;
      private String sFilePath;
      FormUpdating frmUpdate = null;
      private BackgroundWorker bw;
      private String sFileTemplate = "*.?.??.idx";

      public MainForm()
      {
         InitializeComponent();
         FWListBox.SelectedPath = Directory.GetCurrentDirectory();
      }

      #region Events
      private void MainForm_Load(object sender, EventArgs e)
      {
         bUpdateInProgress = false;

         InitAPI();

         if (FWListBox.Items.Count > 0)
            FWListBox.SelectedIndex = 0;
      }

      /// <summary>
      /// Force operator to wait before exit
      /// </summary>
      /// <param name="sender"></param>
      /// <param name="e"></param>
      private void MainForm_FormClosing(object sender, FormClosingEventArgs e)
      {
         if (bUpdateInProgress)
         {
            //MessageBox.Show("Update in Progress - Please wait before closing application");
            e.Cancel = true;
         }
         else if (tpApi != null)
            tpApi.TpApi_TerminateAPI();
      }

      private void ExitBtn_Click(object sender, EventArgs e)
      {
         Close();
      }

      /// <summary>
      /// The device selection has changed.  Close the current
      /// device and open the newly selected one.
      /// </summary>
      /// <param name="sender"></param>
      /// <param name="e"></param>
      private void devCombo_SelectedIndexChanged(object sender, EventArgs e)
      {
         UseWaitCursor = true;
         if (tpDevice != null)
            tpDevice.CloseDevice();

         FWListBox.Pattern = "*.?.??.idx";
         if (devCombo.Items.Count > 0)
         {
            tpDevice = tpDevices[devCombo.SelectedIndex];
            
            // Convert device info to Strings
            String strDevName;
            strDevName = "";
            TpApi.s_regDevInfo devInfo = tpDevice.GetRegDevInfo();
            for (int i = 0; devInfo.device_name[i] != 0; i++)
               strDevName = strDevName + devInfo.device_name[i];
            UpdateLabel.Text = strDevName;


            TpApiError r = tpDevice.OpenDevice();
            if ((r == TpApiError.TPAPI_NOERROR) && tpDevice.DeviceConnected())
            {
               if (GetVersion(tpDevice))
                  StartUpdateBtn.Enabled = true;
            }
            else
            {
               //UpdateLabel.Text = "No Device Found";
               StartUpdateBtn.Enabled = false;
            }
         }
         else
            StartUpdateBtn.Enabled = false;

         UseWaitCursor = false;
      }

      /// <summary>
      /// Initiate a firmware update using the firmware file
      /// selected in the fileview window
      /// </summary>
      /// <param name="sender"></param>
      /// <param name="e"></param>
      private void StartUpdateBtn_Click(object sender, EventArgs e)
      {
         // Test if the file exists
         sFilePath = FWListBox.SelectedFile;
         if (sFilePath == null) return;

         if (MessageBox.Show(this, "Do not disconnect or otherwise interrupt the "
            + "update process,  \r\nas this could render the device inoperable.  "
            + "\r\n\nContinue with update?", "Firmware Update", MessageBoxButtons.OKCancel,
            MessageBoxIcon.Warning) != System.Windows.Forms.DialogResult.OK)
            return;

         bUpdateInProgress = true;
         StartUpdateBtn.Enabled = false;
         ExitBtn.Enabled = false;
         UseWaitCursor = true;

         // Kick off worker thread
         bw = new BackgroundWorker();
         bw.WorkerReportsProgress = true;
         bw.WorkerSupportsCancellation = false;

         bw.DoWork += bw_DoWork;
         bw.ProgressChanged += bw_ProgressChanged;
         bw.RunWorkerCompleted += bw_RunWorkerCompleted;

         bw.RunWorkerAsync("Hello to worker");
      }
      #endregion

      void bw_DoWork(object sender, DoWorkEventArgs e)
      {
         try {
            // Open and read in firmware data
            byte[] bData = ReadFile(sFilePath);

            bw.ReportProgress(0);
            TpApiError r = tpDevice.SendBulkData(
               TpApiDevice.eScnDataType.SCN_DTYPE_DSP, ref bData);
            if (r == TpApiError.TPAPI_NOERROR)
            {
               bw.ReportProgress(50);
               tpDevice.ResetDevice();
            }
            e.Result = r;    // This gets passed to RunWorkerCompleted
         }
         catch (Exception ex)
         {
            e.Result = ex;
         }
      }

      void bw_RunWorkerCompleted(object sender, RunWorkerCompletedEventArgs e)
      {
         if (!e.Cancelled && (e.Error == null) &&
            (e.Result.ToString() == TpApiError.TPAPI_NOERROR.ToString()))
         {
            GetVersion(tpDevice);
            frmUpdate.label1.Location = new Point(60, 16);
            frmUpdate.label1.Text = "Update Successful";
            System.Media.SystemSounds.Beep.Play();
         }
         else
         {
            frmUpdate.label1.Location = new Point(55, 16);
            frmUpdate.label1.Text = e.Result.ToString();
            frmUpdate.label1.Font = new Font("Calibri", 12F, System.Drawing.FontStyle.Regular, System.Drawing.GraphicsUnit.Point, ((byte)(0)));
            frmUpdate.pBoxInfo.Image = global::FirmwareUpdate.Properties.Resources.error;
            System.Media.SystemSounds.Exclamation.Play();
         }
         frmUpdate.Text = " Update Complete";
         frmUpdate.bUpdateInProgress = false;
         frmUpdate.UseWaitCursor = false;
         frmUpdate.Cursor = Cursors.Default;
         frmUpdate.pBoxProgBar.Visible = false;
         frmUpdate.label2.Visible = false;
         frmUpdate.pBoxInfo.Visible = true;
         frmUpdate.btnExit.Visible = true;
         frmUpdate.Refresh();

         bUpdateInProgress = false;
         StartUpdateBtn.Enabled = true;
         ExitBtn.Enabled = true;
         UseWaitCursor = false;
      }

      void bw_ProgressChanged(object sender, ProgressChangedEventArgs e)
      {
         if (e.ProgressPercentage == 0)
         {
            frmUpdate = new FormUpdating();
            frmUpdate.Location = new Point(this.Location.X + 63, this.Location.Y + 116);
            frmUpdate.label1.Location = new Point(16, 16);
            frmUpdate.pBoxInfo.Visible = false;
            frmUpdate.pBoxProgBar.Visible = true;
            frmUpdate.ShowDialog();
         }
         else if (e.ProgressPercentage == 50)
         {
            frmUpdate.Text = " Reset Device";
            frmUpdate.label1.Text = "Resetting the Device...";
            frmUpdate.label2.Text = "This may take up to 15 seconds.  Please wait.";
            frmUpdate.Refresh();
         }
      }

      private static byte[] ReadFile(string filePath)
      {
         byte[] buffer;
         FileStream fileStream = new FileStream(filePath, FileMode.Open, FileAccess.Read);
         try
         {
           int length = (int)fileStream.Length;  // get file length
           buffer = new byte[length];            // create buffer
           int count;                            // actual number of bytes read
           int sum = 0;                          // total number of bytes read

           // read until Read method returns 0 (end of the stream has been reached)
           while ((count = fileStream.Read(buffer, sum, length - sum)) > 0)
             sum += count;  // sum is a buffer offset for next reading
         }
         finally
         {
           fileStream.Close();
         }
         return buffer;
      }

      /// <summary>
      /// Retrieve the recommended firmware version and filename
      /// information for the attached device
      /// </summary>
      private bool ParseManifest(String sModel, ref String sVer, ref String sFile)
      {
         XmlDocument Doc = new XmlDocument();
         String fileName = "FirmwareManifest.xml";
         if (File.Exists(fileName))
         try {
            using (XmlReader Reader = new XmlTextReader(fileName))
            {
               Doc.Load(Reader);
               foreach (XmlNode node in Doc.DocumentElement.ChildNodes)
               {
                  if (node.HasChildNodes)
                  {
                     foreach (XmlNode childNode in node.ChildNodes)
                     {
                        if ((childNode.Attributes.Count == 1) &&
                            (childNode.Attributes[0].Value == sModel))
                        {
                           if (childNode.ChildNodes.Count == 2)
                           {
                              if (childNode.ChildNodes[0].Name == "FirmwareVersion")
                                 sVer = childNode.ChildNodes[0].InnerText;
                              if (childNode.ChildNodes[1].Name == "FirmwareFile")
                                 sFile = childNode.ChildNodes[1].InnerText;
                              return true;
                           }
                        }
                     }
                  }
               }
            }
         }
         catch (Exception e)
         {
            MessageBox.Show("Parse Manifest error: " + e.ToString());
         }
         return false;
      }

      /// <summary>
      /// Retrieve and display the model and firmware
      /// version information for the attached device
      /// 
      /// Display only the firmware files applicable to
      /// the attached device
      /// </summary>
      /// <returns></returns>
      private bool GetVersion(TpApiDevice dev)
      {
         TpApiDevice.s_devInfo devInfo = new TpApiDevice.s_devInfo();

         // Retrieve model name and firmware version from device
         TpApiError r = dev.GetDeviceInfo(ref devInfo);
         if (r == TpApiError.TPAPI_NOERROR)
         {
            // Convert device info to Strings
            String strModel = "";
            for (int i = 0; devInfo.assy_model[i] != 0; i++)
               strModel = strModel + devInfo.assy_model[i];

            String strVer = "";
            for (int i = 0; devInfo.sw_version[i] != 0; i++)
               strVer = strVer + devInfo.sw_version[i];
            VerLabel.Text = strVer;

            // Parse the manifest file for the recommended
            // firmware version and filename
            String sRecVer = ""; String sFile = "";
            if (ParseManifest(strModel, ref sRecVer, ref sFile))
            {
               
               RecVersLabel.Text = sRecVer;
               RecFileLabel.Text = sFile;
      
               // Trim off the version and extension, add wildcards
               sFile = sFile.Substring(0, sFile.Length - 8);
               FWListBox.Pattern = sFile + "?.??.idx";
      
               // Determine whether an update is necessary
               if (RecVersLabel.Text == strVer)
               {
                  UpdateLabel.Text = "No Update is Necessary";
                  UpdateLabel.ForeColor = SystemColors.ControlText;
                  UpdateLabel.BackColor = SystemColors.Control;
               }
               else
               {
                  UpdateLabel.Text = "An Update is Recommended";
                  UpdateLabel.ForeColor = SystemColors.ActiveCaptionText;
                  UpdateLabel.BackColor = SystemColors.Highlight;
               }
            }
            else
            {
               UpdateLabel.Text = "No Recommendation Available";
               UpdateLabel.ForeColor = SystemColors.ControlText;
               UpdateLabel.BackColor = SystemColors.Control;
               FWListBox.Pattern = "*.?.??.idx";
            }
            // Update file list & selection only if necessary
            if (FWListBox.Pattern != sFileTemplate)
            {
               sFileTemplate = FWListBox.Pattern;
               FWListBox.Refresh();
               if (FWListBox.Items.Count > 0)
                  FWListBox.SelectedIndex = 0;
            }
            return true;
         }
         else
         {
            MessageBox.Show("GetVersion Failed" + r.ToString());
            return false;
         }
      }

      /// <summary>
      /// Retrieve and display all available devices
      /// </summary>
      /// <returns></returns>
      private TpApiError GetDeviceList()
      {
         TpApiError r;
         uint numDev = MAX_DEVLIST;
         tpDevices = new TpApiDevice[MAX_DEVLIST];

         r = tpApi.TpApi_GetDeviceList(ref numDev, ref tpDevices);
         if (r == TpApiError.TPAPI_NOERROR)
         {
            if (numDev > 0)
            {
               devCombo.Items.Clear();
               for (int ii = 0; ii < numDev; ii++)
               {
                  // Convert device info to Strings
                  String strModel, strSerNo, strLnk;
                  strModel = "";
                  strSerNo = "";
                  strLnk = "";
                  TpApi.s_regDevInfo devInfo = tpDevices[ii].GetRegDevInfo();

                  for (int i = 0; devInfo.assy_model[i] != 0; i++)
                     strModel = strModel + devInfo.assy_model[i];

                  for (int i = 0; devInfo.assy_sn[i] != 0; i++)
                     strSerNo = strSerNo + devInfo.assy_sn[i];

                  switch (devInfo.libType)
                  {
                     case TPAPI_LNK_USB:
                        strLnk = "USB 2.0";
                        break;
                     case TPAPI_LNK_1394_OHCI:
                        strLnk = "IEEE1394 OHCI";
                        break;
                     case TPAPI_LNK_1394_UB:
                        strLnk = "IEEE1394 Unibrain";
                        break;
                     default:
                        strLnk = "Unknown";
                        break;
                  }

                  // Display device info in ComboBox
                  devCombo.Items.Add("  " + strModel + " - S/N " +
                     strSerNo + " - " + strLnk);
               }
            }
            else r = TpApiError.TPAPI_DEVICEUNAVAILABLE;
         }
         else MessageBox.Show("GetDeviceList Failed " + r.ToString());

         if (devCombo.Items.Count > 0)
            devCombo.SelectedIndex = 0;
         return r;
      }

      /// <summary>
      /// Initializes the TPAPI library and attempts to establish
      /// communication with the scanner.  This will not reset, or
      /// otherwise alter the state of the scanner.
      /// </summary>
      private void InitAPI()
      {
         UseWaitCursor = true;

         FWListBox.Pattern = "*.?.??.idx";
         StartUpdateBtn.Enabled = false;
         ExitBtn.Enabled = true;
         UpdateLabel.Text = "Initializing...";
         UpdateLabel.Refresh();

         // Initialize the TP API library
         TpApiError r = tpApi.TpApi_InitializeApi();
         if (r != TpApiError.TPAPI_NOERROR)
            UpdateLabel.Text = r.ToString();
         else
         {  // Find all attached devices and update the device combo box
            r = GetDeviceList();
         }

         UseWaitCursor = false;
         if (r != TpApiError.TPAPI_NOERROR)
         {
            StartUpdateBtn.Enabled = false;
            if (r == TpApiError.TPAPI_DEVICEUNAVAILABLE)
               UpdateLabel.Text = "No Device Found";
            else
               UpdateLabel.Text = r.ToString();
         }
      }
   }
}
